#include <stdio.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_math.h>
#include <gsl/gsl_roots.h>

#include "quadratic.h"

#define MAX_ITERATIONS 100

int main()
{
    int status;
    int i;
    const gsl_root_fdfsolver_type *solver_type;
    gsl_root_fdfsolver *solver;
    gsl_function_fdf FDF;
    double x0;

    /* We want to solve x^2 - 5 */
    struct quadratic_params params = { 1.0, 0.0, -5.0 };
    double r_expected = sqrt(5.0);
    /* Starting from a guess of 5.0 */
    double x = 5;

    /* Set up the function */
    FDF.f = &quadratic;
    FDF.df = &quadratic_deriv;
    FDF.fdf = &quadratic_fdf;
    FDF.params = &params;

    /* Allocate a newton solver and set it to use FDF */
    solver_type = gsl_root_fdfsolver_newton;
    solver = gsl_root_fdfsolver_alloc(solver_type);
    gsl_root_fdfsolver_set(solver, &FDF, x);

    printf("using %s method\n", gsl_root_fdfsolver_name(solver));

    printf ("%-5s %10s %10s %10s\n", "iter", "root", "err", "err(est)");

    status = GSL_CONTINUE;
    for (i = 1; i <= MAX_ITERATIONS && status == GSL_CONTINUE; ++i) {
        /* Iterate one step of the solver */
        status = gsl_root_fdfsolver_iterate(solver);
        if (status != GSL_SUCCESS)
            break;

        /* Get the new approximate solution */
        x0 = x;
        x = gsl_root_fdfsolver_root(solver);

        /* Check to see if the solution is within 0.001 */
        status = gsl_root_test_delta(x, x0, 0, 0.001);
        if (status == GSL_SUCCESS)
            printf("Converged:\n");

        printf ("%5d %10.7f %+10.7f %10.7f\n", i, x, x - r_expected, x - x0);
    }

    /* Free the solver */
    gsl_root_fdfsolver_free(solver);

    if (status == GSL_CONTINUE) {
        printf("error: too many iterations");
    } else if (status != GSL_SUCCESS) {
        printf("error: %s\n", gsl_strerror(status));
    }

    return status;
}

